*******************************************************************************
*
* SYSTEM:  HOT LOCKS
* TITLE:   PARSE_TO_RDB_RSB_CHAIN
*
* OVERVIEW:		Revised : <DD-MMM-YYYY> By <Programmer/Analyst>
* 	Reads output from RSB_CHAIN.EXE, ignoring other lines in the batch
*	log file, and placing results into database HOT_LOCKS.RDB.  It is
*	assumed that there is at least one 'SHOW TIME' or record in the log
*	file that contains a time stamp.
*
* PARAMETERS:
*	This program expects a filename, including wild cards, as input.
*
* MOD:	<DD-MMM-YYYY>	<ISR# MOD# >	<USER_ID>
* 	<Description... Keep this until first modification>
*
* RFILE:	SYS_HOT_FIL_01_*.LOG
* RDESC:	Output log file that ran RSB_CHAIN.EXE.
*
* MFILE:	HOT_LOCKS.RDB
* MDESC:	RDB database containing tables FILE_NODE_LOCKS and NODES.
*
*******************************************************************************

IDENTIFICATION DIVISION.
PROGRAM-ID.   PARSE_TO_RDB_RSB_CHAIN.
AUTHOR.       Garry Fruth.
DATE-WRITTEN. 25-MAR-1996.

ENVIRONMENT DIVISION.
CONFIGURATION SECTION.
SOURCE-COMPUTER. VMS.
OBJECT-COMPUTER. VMS.
SPECIAL-NAMES. CONSOLE IS SYS$ERROR.

DATA DIVISION.
WORKING-STORAGE SECTION.
*temporary variables; generally used within a single paragraph
01  TEMP-NUM				PIC S9(9) COMP.
01  VMS-TIME-ASC			PIC X(23).

*lib$find_file variables
01  FIND_FILE_CONTEXT			PIC S9(9) COMP.
01  INPUT-FILE-SPEC			PIC X(128).
01  DEFAULT-FILE-SPEC			PIC X(26) 
			VALUE "LOG:SYS_HOT_FIL_01*.LOG;*".
01  RMS$_NMF				PIC S9(9) COMP VALUE EXTERNAL RMS$_NMF.

*parameters to parse_rsb_chain
01  PARSE_FILE_NAME			PIC X(128) VALUE SPACES.
01  PARSE_RESULTS.
    05  PARSE-STATUS			PIC S9(9) COMP.
        88  PARSE-SUCCESS			VALUE 1.
    05  PARSE-MSG			PIC X(84).
    05  VMS_TIME_STAMP			SQL_DATE_VMS.
        88  NO-VMS-TIME-STAMP			VALUE 0.
    05  LOCAL_NODE_NAME			PIC X(8).
    05  TOTAL_LOCKING			PIC S9(9) COMP.
    05  TOTAL_LOCAL_LOCKING		PIC S9(9) COMP.
    05  TOTAL_DIST-LOCKING		PIC S9(9) COMP.
    05  LOCK-ACTIVITY-RECORDS		PIC S9(9) COMP.
    05  LOCK-ACTIVITY
		OCCURS 0 TO 200 TIMES DEPENDING ON LOCK-ACTIVITY-RECORDS.
        10  LOCK-DESC			PIC X(128).
        10  LOCK-DESC-LEN		PIC S9(9) COMP.
        10  MASTER-NODE           	PIC X(8).
        10  LOCAL-LOCKING		PIC S9(9) COMP.
        10  DIST-LOCKING		PIC S9(9) COMP.

*sql work variables
01  STORES_OUTSTANDING			PIC S9(9) COMP VALUE 0.
01  LOCK-ACTIVITY-1.
        10  LOCAL-LOCKING		PIC S9(9) COMP.
        10  DIST-LOCKING		PIC S9(9) COMP.
        10  FILENAME.
            49  LOCK-DESC-LEN		PIC S9(4) COMP.
            49  LOCK-DESC		PIC X(128).

EXEC SQL INCLUDE SQLCA END-EXEC.

EXEC SQL DECLARE ALIAS FILENAME 'HOT_LOCKS' END-EXEC.


PROCEDURE DIVISION.
A000_MAIN_LOGIC.
	PERFORM READY-RDB.

	DISPLAY "Enter file spec, wildcards allowed, ["
		DEFAULT-FILE-SPEC
		"]: " WITH NO ADVANCING
	ACCEPT INPUT-FILE-SPEC

	PERFORM FIND-FILE
	PERFORM PARSE_FILES
	  UNTIL PARSE-FILE-NAME = SPACES.

	IF STORES_OUTSTANDING > 0
		PERFORM END-TRANS.

	STOP RUN.

PARSE_FILES.
	PERFORM VARYING TEMP-NUM FROM 1 BY 1
		  UNTIL PARSE-FILE-NAME(TEMP-NUM:1) = SPACE
		CONTINUE
	END-PERFORM
	DISPLAY "Processing ", PARSE-FILE-NAME (1:TEMP-NUM) WITH NO ADVANCING

	CALL "PARSE_RSB_CHAIN"
		USING PARSE_FILE_NAME, PARSE-RESULTS.
	IF PARSE-SUCCESS
*		PERFORM DUMP-PARSE-RESULTS
		PERFORM STORE-PARSE-RESULTS.

	IF PARSE-SUCCESS
		DISPLAY " Success"
	ELSE
		DISPLAY " Failure: ", PARSE-MSG, PARSE-STATUS WITH CONVERSION.

	PERFORM FIND-FILE.

FIND-FILE.
	CALL "LIB$FIND_FILE" USING
		BY DESCRIPTOR	INPUT-FILE-SPEC
				PARSE-FILE-NAME
		BY REFERENCE	FIND-FILE-CONTEXT
		BY DESCRIPTOR	DEFAULT-FILE-SPEC
		GIVING TEMP-NUM.
	IF TEMP-NUM = RMS$_NMF
		MOVE SPACES TO PARSE-FILE-NAME
	ELSE
	IF TEMP-NUM FAILURE
		CALL "LIB$SIGNAL" USING BY VALUE TEMP-NUM.

DUMP-PARSE-RESULTS.
	MOVE SPACES TO VMS-TIME-ASC
	CALL "SYS$ASCTIM" USING
		OMITTED, BY DESCRIPTOR VMS-TIME-ASC, BY REFERENCE VMS-TIME-STAMP
	DISPLAY "PARSE-RESULTS:"
	DISPLAY "	VMS-TIME:		", VMS-TIME-ASC," ",VMS-TIME-STAMP WITH CONVERSION
	DISPLAY "	LOCAL NODE NAME:	", LOCAL-NODE-NAME
	DISPLAY "	TOTAL LOCKING:		", TOTAL-LOCKING WITH CONVERSION
	DISPLAY "	      LOCAL:		", TOTAL-LOCAL-LOCKING WITH CONVERSION
	DISPLAY "	      DIST:		", TOTAL-DIST-LOCKING WITH CONVERSION
	DISPLAY "	LOCK RECORDS:		", LOCK-ACTIVITY-RECORDS WITH CONVERSION
	DISPLAY "	NODE          LOCAL       DIST DESC"
	PERFORM DUMP-PARSE-RESULTS-ACTIVITY
	VARYING TEMP-NUM FROM 1 BY 1 UNTIL TEMP-NUM > LOCK-ACTIVITY-RECORDS.

DUMP-PARSE-RESULTS-ACTIVITY.
	DISPLAY "	",
		MASTER-NODE IN LOCK-ACTIVITY (TEMP-NUM)
		" ",
		LOCAL-LOCKING IN LOCK-ACTIVITY (TEMP-NUM) WITH CONVERSION
		" ",
		DIST-LOCKING IN LOCK-ACTIVITY (TEMP-NUM) WITH CONVERSION
		" ",
		LOCK-DESC IN LOCK-ACTIVITY (TEMP-NUM) 
			(1: LOCK-DESC-LEN IN LOCK-ACTIVITY (TEMP-NUM)).

STORE-PARSE-RESULTS.
	PERFORM START-TRANS

	EXEC SQL INSERT INTO 
		NODES	(TMSTMP, NODE, LOCAL_LOCKING, DIST_LOCKING)
		VALUES	(:PARSE_RESULTS.VMS_TIME_STAMP, 
			 :PARSE_RESULTS.LOCAL_NODE_NAME, 
			 :PARSE_RESULTS.TOTAL_LOCAL_LOCKING,
			 :PARSE_RESULTS.TOTAL_DIST_LOCKING)
	END-EXEC.

	PERFORM STORE-LOCK-ACTIVITY
	VARYING TEMP-NUM FROM 1 BY 1
	  UNTIL TEMP-NUM > LOCK-ACTIVITY-RECORDS
	     OR SQLCODE NOT = 0

	IF SQLCODE = -1001
		MOVE "Duplicate entry or other integrity constraint failed"
		  TO PARSE-MSG
		MOVE SQLCODE TO PARSE_STATUS
		CALL "SQL$SIGNAL"
*		EXEC SQL ROLLBACK END-EXEC
	ELSE
	IF SQLCODE NOT = 0 
		CALL "SQL$SIGNAL"
		DISPLAY "STOP RUN IN STORE-PARSE-RESULTS"
		STOP RUN.

	IF STORES_OUTSTANDING > 99
		PERFORM END-TRANS.

STORE-LOCK-ACTIVITY.
*Store a lock activity record associated with a file on the local node

*	occurs not supported in sql, use temporary data structure
	MOVE CORR LOCK_ACTIVITY(TEMP_NUM) TO LOCK-ACTIVITY-1
	MOVE CORR LOCK_ACTIVITY(TEMP_NUM) TO FILENAME OF LOCK-ACTIVITY-1

	IF MASTER_NODE IN LOCK_ACTIVITY(TEMP_NUM) = LOCAL_NODE_NAME
*		file is master locally on this node
		EXEC SQL INSERT INTO
		   FILE_NODE_LOCKS (TMSTMP, NODE, LOCAL_LOCKING, FILENAME)
		   VALUES (:PARSE_RESULTS.VMS_TIME_STAMP, 
			    PARSE_RESULTS.LOCAL_NODE_NAME,
			    LOCK_ACTIVITY_1.LOCAL_LOCKING,
			    LOCK_ACTIVITY_1.FILENAME)
		END_EXEC
	ELSE
*		file is master on another cluster member
		EXEC SQL INSERT INTO
		   FILE_NODE_LOCKS (TMSTMP, NODE, DIST_LOCKING, FILENAME)
		   VALUES (:PARSE_RESULTS.VMS_TIME_STAMP, 
			    PARSE_RESULTS.LOCAL_NODE_NAME,
			    LOCK_ACTIVITY_1.DIST_LOCKING,
			    LOCK_ACTIVITY_1.FILENAME)
		END_EXEC.

READY-RDB.

START-TRANS.
	IF STORES_OUTSTANDING = 0
		EXEC SQL SET TRANSACTION READ WRITE WAIT 1000 
			EVALUATING FILE_NODE_LOCKS_PRIMARY1 AT VERB TIME
			RESERVING NODES, FILE_NODE_LOCKS FOR PROTECTED WRITE
			END-EXEC
		IF SQLCODE NOT = 0 
			CALL "SQL$SIGNAL"
			DISPLAY "STOP RUN IN START-TRANS"
			STOP RUN
		ELSE
			MOVE 1 TO STORES_OUTSTANDING
	ELSE
		ADD 1 TO STORES_OUTSTANDING.

START-BATCH-TRANS.
	IF STORES_OUTSTANDING = 0
		EXEC SQL SET TRANSACTION BATCH UPDATE END-EXEC
		IF SQLCODE NOT = 0 
			CALL "SQL$SIGNAL"
			DISPLAY "STOP RUN IN START-BATCH-TRANS"
			STOP RUN
		ELSE
			MOVE 1 TO STORES_OUTSTANDING
	ELSE
		ADD 1 TO STORES_OUTSTANDING.

END-TRANS.
	EXEC SQL COMMIT END-EXEC.
	IF SQLCODE NOT = 0 
		CALL "SQL$SIGNAL"
		DISPLAY "STOP RUN IN END-TRANS"
		STOP RUN
	ELSE
		MOVE 0 TO STORES_OUTSTANDING.

IDENTIFICATION DIVISION.
PROGRAM-ID.   PARSE_RSB_CHAIN.
ENVIRONMENT DIVISION.
INPUT-OUTPUT SECTION.
FILE-CONTROL.
    SELECT INPUT_FILE
        ASSIGN TO ".LOG".
DATA DIVISION.

FILE SECTION.

FD INPUT_FILE
   VALUE OF ID IS INPUT_LOG_FILE_WS
   RECORD VARYING FROM 0 TO 256 CHARACTERS DEPENDING ON INPUT_RECORD_SIZE.
01 INPUT_RECORD.
   10  INPUT-RECORD-CHAR	PIC X
		OCCURS 0 TO 256 TIMES DEPENDING ON INPUT_RECORD_SIZE.
01 INPUT_RECORD_20		PIC X(20).
*RMS$......SYS_DSK5    ...       HNE         13      0      1 93444CC0             SYS_DSK5:[MB$.MR.DB]MRMAILBOX.DAT;2101
01 INPUT-LOCK-RECORD.
   10  FILLER			PIC X(33).
   10  MASTER_NODE		PIC X(7).
   10  OACT			PIC X(7).
   10  FILLER			PIC X(36).
   10  LOCK-DESC		PIC X(128).
*      Total locking:  TOTAL_OLD_ACT=   3400  TOTAL_NEW_ACT=    247
*Total local locking:  TOTAL_OLD_ACT=   2399  TOTAL_NEW_ACT=    119
*Distributed locking:  TOTAL_OLD_ACT=   1001  TOTAL_NEW_ACT=    128
01  INPUT-TOTAL-RECORD.
    10  FILLER			PIC X(36).
    10  TOTAL_OACT		PIC X(7).

WORKING-STORAGE SECTION.
*INPUT_FILE VARIABLES
01  INPUT_LOG_FILE_WS		PIC X(128).
01  INPUT_RECORD_SIZE		PIC  9(9) COMP.
01  FILLER			PIC S9(9) COMP.
    88  NOT-END-OF-INPUT-FILE		VALUE 0.
    88  END-OF-INPUT-FILE		VALUE -1.

*PARSE VARIABLES
01  RETURN-STATUS		PIC S9(9) COMP.
01  TEMP-NUM			PIC S9(9) COMP.
01  PARSE-STATE			PIC S9(9) COMP.
    88  INIT-STATE			VALUE 10, 11.
    88  INIT-STATE-1			VALUE 10.
    88  INIT-STATE-2			VALUE 11.
    88  HEADER-STATE			VALUE 12.
    88  LOCAL-LOCK-STATE-1		VALUE 21.
    88  LOCAL-LOCK-STATE-N		VALUE 22.
    88  DIST-LOCK-STATE-1		VALUE 31.
    88  DIST-LOCK-STATE-2		VALUE 32.
    88  DIST-LOCK-STATE-N		VALUE 33.
    88  DIST-LOCK-STATE			VALUE 31 THRU 33.
    88  GRAND-STATE			VALUE 41 THRU 43.
    88  GRAND-STATE-1			VALUE 41.
    88  GRAND-STATE-2			VALUE 42.
    88  GRAND-STATE-3			VALUE 43.
    88  END-STATE			VALUE 51, 52.
    88  END-STATE-1			VALUE 51.
    88  END-STATE-2			VALUE 52.
    88  NON-PROCESS-STATE		VALUE -1, 10 THRU 12, 31, 32, 51, 52.
    88  FAIL-STATE			VALUE -1.
01  FILLER.
      05  TIME-STAMP-ASC.
          10  TIME-STAMP-ASC-1		PIC X(11).
          10  TIME-STAMP-ASC-2		PIC X(1).
          10  TIME-STAMP-ASC-3		PIC X(8).

LINKAGE SECTION.
  01  INPUT_LOG_FILE			PIC X(128).
* output
  01  PARSE-RESULTS.
      05  PARSE-STATUS			PIC S9(9) COMP.
          88  PARSE-SUCCESS			VALUE 1.
      05  PARSE-MSG			PIC X(84).
      05  VMS-TIME-STAMP		SQL_DATE_VMS.
          88  NO-VMS-TIME-STAMP			VALUE 0.
      05  LOCAL_NODE_NAME		PIC X(8).
      05  TOTAL_LOCKING			PIC S9(9) COMP.
      05  TOTAL_LOCAL_LOCKING		PIC S9(9) COMP.
      05  TOTAL_DIST_LOCKING		PIC S9(9) COMP.
      05  LOCK-ACTIVITY-RECORDS		PIC S9(9) COMP.
      05  LOCK-ACTIVITY
		OCCURS 0 TO 200 TIMES DEPENDING ON LOCK-ACTIVITY-RECORDS.
          10  LOCK-DESC			PIC X(128).
          10  LOCK-DESC-LEN		PIC S9(9) COMP.
          10  MASTER-NODE               PIC X(8).
          10  LOCAL-LOCKING		PIC S9(9) COMP.
          10  DIST-LOCKING		PIC S9(9) COMP.

*for parse_rsb_chain subroutine
PROCEDURE DIVISION USING INPUT_LOG_FILE, PARSE-RESULTS.
DECLARATIVES.
INPUT_FILE_ERROR SECTION. USE AFTER STANDARD ERROR PROCEDURE ON INPUT_FILE.
INPUT_FILE_ERROR_1.
	SET FAIL-STATE TO TRUE.
	SET END-OF-INPUT-FILE TO TRUE.
END DECLARATIVES.
MAIN_PROGRAM SECTION.
A000_MAIN_LOGIC.
*	initialize subroutine
	MOVE SPACES TO PARSE-MSG
	MOVE 0 TO PARSE-STATUS
	MOVE INPUT_LOG_FILE TO INPUT_LOG_FILE_WS
	SET NO-VMS-TIME-STAMP TO TRUE
	SET INIT-STATE-1 TO TRUE
	MOVE 0 TO LOCK-ACTIVITY-RECORDS
	SET NOT-END-OF-INPUT-FILE TO TRUE
	OPEN INPUT INPUT_FILE
	IF FAIL-STATE
		DISPLAY "Error opening input file " INPUT_LOG_FILE
			RMS-STS OF INPUT_FILE WITH CONVERSION
			RMS-STV OF INPUT_FILE WITH CONVERSION.

*	process records
	PERFORM READ-INPUT-FILE
	PERFORM PARSE-INPUT-FILE
	  UNTIL END-OF-INPUT-FILE OR END-STATE-2 OR FAIL-STATE.

*	exit
	CLOSE INPUT_FILE
	IF NO-VMS-TIME-STAMP
		MOVE "No timestamp found" TO PARSE-MSG
		MOVE 388 TO PARSE-STATUS
	ELSE
	IF FAIL-STATE
		MOVE 44 TO PARSE-STATUS
	ELSE
	IF END-STATE
		SET PARSE-SUCCESS TO TRUE
	ELSE
	IF PARSE-MSG = SPACES
		MOVE "Exited parse without being in end-state?" TO PARSE-MSG
		MOVE 44 TO PARSE-STATUS.

	
	EXIT PROGRAM.

PARSE-INPUT-FILE.
*Input
*	state and input-record
*process
*	Parse record to determine new state
*	Process record based on state
*	Get next record
*output
*	state and input-record or eof

*	Determine state transition
	IF INIT-STATE-1 OR END-STATE-1
		IF NO-VMS-TIME-STAMP
			PERFORM PARSE-TIME-STAMP
		END-IF
		IF NOT NO-VMS-TIME-STAMP
			ADD 1 TO PARSE-STATE.

	EVALUATE TRUE ALSO INPUT-RECORD-20
	WHEN INIT-STATE			ALSO "Resource name"
		SET HEADER-STATE TO TRUE
	WHEN INIT-STATE			ALSO ANY
		MOVE "Looking for 'Resource name'" TO PARSE-MSG
	WHEN HEADER-STATE		ALSO ANY
		MOVE "Parsing first local lock record" TO PARSE-MSG
		SET LOCAL-LOCK-STATE-1 TO TRUE
	WHEN LOCAL-LOCK-STATE-1		ALSO ANY
		MOVE "Looking for blank line to end local locks" TO PARSE-MSG
		SET LOCAL-LOCK-STATE-N TO TRUE
	WHEN LOCAL-LOCK-STATE-N		ALSO SPACES
		MOVE "Looking for --distributed locks-" TO PARSE-MSG
		SET DIST-LOCK-STATE-1 TO TRUE
	WHEN DIST-LOCK-STATE-1		ALSO "--distributed locks-"
		MOVE "Looking for '      Total locking:" TO PARSE-MSG
		SET DIST-LOCK-STATE-2 TO TRUE
	WHEN DIST-LOCK-STATE-1		ALSO ANY
*		must be --distributed locks--
		SET FAIL-STATE TO TRUE
	WHEN DIST-LOCK-STATE-2		ALSO ANY
		SET DIST-LOCK-STATE-N TO TRUE
	WHEN DIST-LOCK-STATE-N		ALSO "      Total locking:"
		MOVE "Looking for 'Total local locking:" TO PARSE-MSG
		SET GRAND-STATE-1 TO TRUE
	WHEN GRAND-STATE-1		ALSO "Total local locking:"
		MOVE "Looking for 'Distributed locking:" TO PARSE-MSG
		SET GRAND-STATE-2 TO TRUE
	WHEN GRAND-STATE-2		ALSO "Distributed locking:"
		SET GRAND-STATE-3 TO TRUE
	WHEN GRAND-STATE-3		ALSO ANY
		IF NO-VMS-TIME-STAMP
			SET END-STATE-1 TO TRUE
		ELSE
			SET END-STATE-2 TO TRUE
		END-IF
	WHEN GRAND-STATE		ALSO ANY
		SET FAIL-STATE TO TRUE.

*	DISPLAY PARSE-STATE WITH CONVERSION,
*		"	", INPUT-RECORD.

*	process record
	EVALUATE TRUE
	WHEN NON-PROCESS-STATE
*		ignore records until we have something to process
		CONTINUE
	WHEN LOCAL-LOCK-STATE-1
*		parse the first local lock record, contains node name
		PERFORM PARSE-LOCAL-LOCK-1
	WHEN LOCAL-LOCK-STATE-N
*		parse remaining local locks, transition if blank line
		PERFORM PARSE-LOCAL-LOCK-N
	WHEN DIST-LOCK-STATE-N
*		parse distributed lock record
		PERFORM PARSE-DIST-LOCK-N
 	WHEN GRAND-STATE
*		Total locking, Total Local Locking or Total Distributed Locking
		PERFORM PARSE-GRAND-TOTALS
	WHEN OTHER
		MOVE "Parse-state invalid" TO PARSE-MSG
		SET FAIL-STATE TO TRUE.

*	get next record
	IF NOT FAIL-STATE
		PERFORM READ-INPUT-FILE.

PARSE-LOCAL-LOCK-1.
*parse the first local lock, contains node name
	PERFORM PARSE-LOCK-RECORD.
	IF RETURN-STATUS FAILURE
		SET FAIL-STATE TO TRUE
	ELSE
		MOVE MASTER-NODE IN LOCK-ACTIVITY (1) TO LOCAL-NODE-NAME.

PARSE-LOCAL-LOCK-N.
*parse remaining local locks
	PERFORM PARSE-LOCK-RECORD.
	IF RETURN-STATUS FAILURE
		SET FAIL-STATE TO TRUE.

PARSE-DIST-LOCK-N.
*parse distributed lock record
	PERFORM PARSE-LOCK-RECORD.
	IF RETURN-STATUS FAILURE
		SET FAIL-STATE TO TRUE.


PARSE-LOCK-RECORD.
	SET RETURN-STATUS TO SUCCESS
	INSPECT OACT IN INPUT-LOCK-RECORD REPLACING LEADING SPACES BY "0"
	IF OACT IN INPUT-LOCK-RECORD NOT NUMERIC
		MOVE "Invalid number in lock record" TO PARSE-MSG
		SET RETURN-STATUS TO FAILURE
	ELSE
	IF DIST-LOCK-STATE
		IF MASTER-NODE IN INPUT-LOCK-RECORD = LOCAL-NODE-NAME
			MOVE "Local node in distributed section" TO PARSE-MSG
			SET RETURN-STATUS TO FAILURE
		END-IF
	ELSE
		IF LOCAL-LOCK-STATE-N AND 
		   MASTER-NODE IN INPUT-LOCK-RECORD NOT = LOCAL-NODE-NAME
			MOVE "Remote node in local lock section" TO PARSE-MSG
			SET RETURN-STATUS TO FAILURE.

	IF RETURN-STATUS SUCCESS
		ADD 1 TO LOCK-ACTIVITY-RECORDS
		MOVE MASTER-NODE IN INPUT-LOCK-RECORD
		  TO MASTER-NODE IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS)

		MOVE LOCK-DESC IN INPUT-LOCK-RECORD
		  TO LOCK-DESC IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS)
*		remove trailing number, eg version number
		SUBTRACT 83 FROM INPUT-RECORD-SIZE GIVING TEMP-NUM
		PERFORM VARYING TEMP-NUM FROM TEMP-NUM BY -1 UNTIL
			LOCK-DESC IN INPUT-LOCK-RECORD (TEMP-NUM:1) NOT NUMERIC
			CONTINUE
		END-PERFORM
		IF LOCK-DESC IN INPUT-LOCK-RECORD (TEMP-NUM:1) = ";"
			SUBTRACT 1 FROM TEMP-NUM
		END-IF
		MOVE TEMP-NUM
		  TO LOCK-DESC-LEN IN LOCK-ACTIVITY(LOCK-ACTIVITY-RECORDS).

	IF DIST-LOCK-STATE AND RETURN-STATUS SUCCESS
		MOVE 0 TO LOCAL-LOCKING IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS)
		MOVE OACT IN INPUT-LOCK-RECORD
		  TO DIST-LOCKING IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS)
	ELSE
		MOVE 0 TO DIST-LOCKING IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS)
		MOVE OACT IN INPUT-LOCK-RECORD
		  TO LOCAL-LOCKING IN LOCK-ACTIVITY (LOCK-ACTIVITY-RECORDS).

SET-LOCAL-NODE.

PARSE-GRAND-TOTALS.
	SET RETURN-STATUS TO SUCCESS
	INSPECT TOTAL_OACT IN INPUT-TOTAL-RECORD REPLACING LEADING SPACES BY "0"
	IF TOTAL_OACT IN INPUT-TOTAL-RECORD NOT NUMERIC
		MOVE "Invalid number in totals record" TO PARSE-MSG
		SET RETURN-STATUS TO FAILURE
		SET FAIL-STATE TO TRUE
	ELSE
	IF GRAND-STATE-1
		MOVE TOTAL-OACT IN INPUT-TOTAL-RECORD TO TOTAL-LOCKING
	ELSE
	IF GRAND-STATE-2
		MOVE TOTAL-OACT IN INPUT-TOTAL-RECORD TO TOTAL-LOCAL-LOCKING
	ELSE
	IF GRAND-STATE-3
		MOVE TOTAL-OACT IN INPUT-TOTAL-RECORD TO TOTAL-DIST-LOCKING
	ELSE
		MOVE "Invalid parse-state?" TO PARSE-MSG
		SET FAIL-STATE TO TRUE
		SET RETURN-STATUS TO FAILURE.

PARSE-TIME-STAMP.
*	put the whole record through $BINTIM and see if it is a time stamp
	IF NO-VMS-TIME-STAMP
		MOVE 1 TO TEMP-NUM
		MOVE SPACES TO TIME-STAMP-ASC
		INSPECT INPUT-RECORD TALLYING TEMP-NUM FOR LEADING SPACES
		UNSTRING INPUT-RECORD DELIMITED BY SPACES
			INTO TIME-STAMP-ASC-1, TIME-STAMP-ASC-3
			WITH POINTER TEMP-NUM
		CALL "SYS$BINTIM" USING
			BY DESCRIPTOR TIME-STAMP-ASC
			BY REFERENCE VMS-TIME-STAMP
			GIVING RETURN-STATUS
		IF RETURN-STATUS FAILURE
			SET NO-VMS-TIME-STAMP TO TRUE
		ELSE
*			round to nearest 15 minutes
			DIVIDE VMS-TIME-STAMP BY 9000000000
				GIVING VMS-TIME-STAMP ROUNDED
			MULTIPLY 9000000000 BY VMS-TIME-STAMP
			CALL "SYS$ASCTIM" USING
				OMITTED
				BY DESCRIPTOR TIME-STAMP-ASC
				BY REFERENCE  VMS-TIME-STAMP
				GIVING RETURN-STATUS.

READ-INPUT-FILE.
	MOVE SPACES TO INPUT-RECORD-20, LOCK-DESC OF INPUT-LOCK-RECORD
	READ INPUT_FILE
		AT END SET END-OF-INPUT-FILE TO TRUE.

END PROGRAM PARSE_RSB_CHAIN.
END PROGRAM PARSE_TO_RDB_RSB_CHAIN.
